#include "DirectionalLight.fxh"
#include "HeightMapHelper.fxh"

float4x4 World;
float4x4 View;
float4x4 Projection;
float3 CameraPosition;

// Used for converting world coordinates to flow map texture coordinates.
float4 FlowMapTexCoordScaleAndOffset;	// (sx, sy, dx, dy)

float NormalTextureScale;
float2 NormalPhase01;		// Phase for normal maps 0 (x) and 1 (y)
float4 NormalOffsets0011;	// (x0, y0, x1, y1)

float DiffuseTextureScale;
float2 DiffusePhase01;		// Phase for diffuse maps 0 (x) and 1 (y)
float4 DiffuseOffsets0011;	// (x0, y0, x1, y1)

// How much can we streeetttchhh the normal maps.
// Higher values allow for faster water motion, but look worse - they create more distortion.
// In general we should keep below 0.5.
float MaxNormalDistortion;
float MaxDiffuseDistortion;
float NormalPulseReduction;
float DiffusePulseReduction;

bool UsePulseReduction;
bool UsingUnsignedFlowMap;

texture NormalTexture0 : register(t0);
sampler2D NormalSampler0 : register(s0) = sampler_state
{
	Texture = <NormalTexture0>;
};

texture NormalTexture1 : register(t1);
sampler2D NormalSampler1 : register(s1) = sampler_state
{
	Texture = <NormalTexture1>;
};

texture FlowMapTexture : register(t2);
sampler2D FlowMapSampler : register(s2) = sampler_state
{
	Texture = <FlowMapTexture>;
};

texture NoiseTexture : register(t3);
sampler2D NoiseSampler : register(s3) = sampler_state
{
	Texture = <NoiseTexture>;
};

// A nice sky for reflecting.
texture SkyTexture : register(t4);
sampler2D SkySampler : register(s4) = sampler_state
{
	Texture = <SkyTexture>;
};
float2 SkyTextureOffset;



bool EnableDiffuse;

texture DiffuseTexture0 : register(t5);
sampler2D DiffuseSampler0 : register(s5) = sampler_state
{
	Texture = <DiffuseTexture0>;
};

texture DiffuseTexture1 : register(t6);
sampler2D DiffuseSampler1 : register(s6) = sampler_state
{
	Texture = <DiffuseTexture1>;
};

// Declare the height map
HEIGHTMAP_REGISTER(0)

// Water points up.
#define WATER_NORMAL float3(0, 1, 0)

// Some constants for the water
#define SPECULAR_POWER 18
#define SPECULAR_INTENSITY 1.1

struct VertexShaderInput
{
    float4 Position : POSITION0;
};

struct VertexShaderOutput
{
    float4 Position : POSITION0;
	float3 WorldPosition : TEXCOORD0;
	float WaterHeight : TEXCOORD1;
};

VertexShaderOutput VertexShaderFunction(VertexShaderInput input)
{
    VertexShaderOutput output;

	// We want to push out 

    float4 worldPosition = mul(input.Position, World);
    float4 viewPosition = mul(worldPosition, View);
    output.Position = mul(viewPosition, Projection);
	output.WorldPosition = worldPosition.xyz;

	output.WaterHeight = GetWorldHeight(worldPosition);

    return output;
}

#define WATER_COLOR float3(0.1, 0.8, 1)
#define LIGHT_COLOR float3(1, 1, 1)

// This points up in the world.
#define FLAT_VECTOR float3(0, 1, 0)

// Used for converting the normal map from uv space to world space.
float3 TerrainNormal = float3(0, 1, 0);
float3 TerrainBinormal = float3(0, 0, -1);
float3 TerrainTangent = float3(1, 0, 0);

// Returns the raw flow vector (max magnitude 1)
void GetFlowVectorAndNoiseSample(float3 worldPosition, out float2 flowVector, out float noiseSample)
{
	// We're using world position to determine where to sample from the flow map, so we'll need to scale and offset it.
	float2 flowTexCoord = (worldPosition.xz + FlowMapTexCoordScaleAndOffset.zw) * FlowMapTexCoordScaleAndOffset.xy;
	flowVector = tex2D(FlowMapSampler, flowTexCoord).rg;

	if (UsingUnsignedFlowMap)
	{
		flowVector = (flowVector * 2) - 1;
	}

	noiseSample = 0;
	if (UsePulseReduction)
	{
		noiseSample = tex2D(NoiseSampler, flowTexCoord).r;
	}
}

// We can't go too high on this, otherwise we get distortions. Parts of the water feel like
// they're picking up speed.
// Another thing to try might be to sample from the noisemap moving along with the normals. That migth be hard though.
#define PULSE_REDUCTION_MAX_EFFECT 0.5

float3 GetNormalFromWaveMap(float3 worldPosition, float2 flowVector, float noiseSample)
{
	float flowStrength = length(flowVector);

	// Too much stretching looks bad.
	flowVector *= MaxNormalDistortion;

	// We base our texture coordinate on where we are in the world.
	float2 texCoord = worldPosition.xz * NormalTextureScale;

	float cycleOffset = noiseSample;
	float phase0 = (NormalPhase01.x + cycleOffset * NormalPulseReduction) % 1;
	float phase1 = (NormalPhase01.y + cycleOffset * NormalPulseReduction) % 1;

	// Sample normal map.  Normals are already in (-1,1) format.
	float3 normal0 =  tex2D(NormalSampler0, texCoord + NormalOffsets0011.xy + flowVector * phase0);
	float3 normal1 = tex2D(NormalSampler1, texCoord + NormalOffsets0011.zw + flowVector * phase1);
	
	// Blend the two. 
	float normal0BlendAmount = 2 * abs(phase0 - 0.5); // This essentially creates a saw wave that determines the blending.
	float3 final = normalize(lerp(normal0, normal1, normal0BlendAmount));

	// Go from tangent space to world space
	float3x3 TangentToWorld = float3x3(TerrainTangent, TerrainBinormal, TerrainNormal);
	final = mul(final, TangentToWorld);

	// Scale down the intensity of the normals for water that isn't moving.
	// REVIEW: This doesn't look that great.
	float3 finalFinal = normalize(lerp(FLAT_VECTOR, final, saturate(flowStrength + 0.3)));
	return finalFinal;
}

float3 CalculateSpecularTerm(float3 worldNormal, float3 worldPosition)
{
	// Calculate specular:
	float3 reflectionVector = normalize(reflect(-DirectionToLight, worldNormal));
	float3 directionToCamera = normalize(CameraPosition - worldPosition);
	float foobar = dot(reflectionVector, directionToCamera);    
	float3 specularTerm = saturate(LIGHT_COLOR * SPECULAR_INTENSITY * pow(saturate(foobar), SPECULAR_POWER));
	return specularTerm;
}

#define PI 3.141592

float3 CalculateReflectionComponent(float3 worldPosition, float3 normal)
{
	// Hacky sky reflection
	float3 skyReflectionVector = normalize(reflect(worldPosition - CameraPosition, normal));

	float2 thetaPhi = ((atan2(skyReflectionVector.yy, skyReflectionVector.xz) + PI) % PI) / PI;
	// Now it should be with 0,1. But this generally just reflects a tiny piece of the sky. Make it bigger.
	thetaPhi *= 3;
	return tex2D(SkySampler, thetaPhi + SkyTextureOffset).rgb;
}

float3 GetDiffuseColor(float3 worldPosition, float2 flowVector, float noiseSample)
{
	if (EnableDiffuse)
	{
		float flowStrength = length(flowVector);

		// Too much stretching looks bad.
		flowVector *= MaxDiffuseDistortion;

		// We base our texture coordinate on where we are in the world.
		float2 texCoord = worldPosition.xz * DiffuseTextureScale;

		float cycleOffset = noiseSample;
		float phase0 = (DiffusePhase01.x + cycleOffset * DiffusePulseReduction) % 1 - 0.5;	// -0.5 Because it's best to distort from -0.5 to 0.5, rather
		float phase1 = (DiffusePhase01.y + cycleOffset * DiffusePulseReduction) % 1 - 0.5;	// than from 0 to 1, like it is with normals (according to Valve).
																							// (I see no difference though)

		// Sample normal map.  Normals are already in (-1,1) format.
		float2 texCoord1 = texCoord + DiffuseOffsets0011.xy + flowVector * phase0;
		float2 texCoord2 = texCoord + DiffuseOffsets0011.zw + flowVector * phase1;
		float3 diffuse0 =  tex2D(DiffuseSampler0, texCoord1);
		float3 diffuse1 = tex2D(DiffuseSampler1, texCoord2);
	
		// Blend the two. phase0 generally goes between -0.5 and 0.5 (this will be a problem when we adjust for NoiseSampler)
		// During this time we want blending to go from 1 to 0 to 1.
		
		float diffuse1BlendAmount = 2 * abs(phase0);
		float3 final = lerp(diffuse0, diffuse1, diffuse1BlendAmount);


		float3 finalFinal = final;
		// TODO: After we get it working, re-enable this! It really should be based on flow strentgh.
		//float3 finalFinal = normalize(lerp(FLAT_VECTOR, final, saturate(flowStrength + 0.3)));

		return finalFinal;		
	}
	return 0;
}

#define DEPTH_FACTOR 2

float4 PixelShaderFunction(VertexShaderOutput input) : COLOR0
{
	float2 flowVector;
	float noiseSample;
	GetFlowVectorAndNoiseSample(input.WorldPosition, flowVector, noiseSample);
	float3 normal = GetNormalFromWaveMap(input.WorldPosition, flowVector, noiseSample);

	// Basic lighting
	float directionalIntensity = saturate(dot(normal, DirectionToLight));
	float3 specularTerm = CalculateSpecularTerm(normal, input.WorldPosition);
	//float3 specularTerm = 0;

	float3 reflectionComponent = CalculateReflectionComponent(input.WorldPosition, normal);

	// I thought this could be useful for "debris", like Valve does. However, it doesn't look great.
	// I don't think it's a good way to do foam, either.
	// It's mainly useful for visualization.
	float3 diffuseColor = GetDiffuseColor(input.WorldPosition, flowVector, noiseSample);

	float3 finalColor = (reflectionComponent + diffuseColor) * directionalIntensity + specularTerm;

	// Test to see how far below the terrain is.
	float waterOpacity = lerp(0.3, 1, saturate((input.WorldPosition.y - input.WaterHeight) * DEPTH_FACTOR));

	// REVIEW: Opacity probably shouldn't affect specular!
	return float4(finalColor, waterOpacity);
}

float4 PixelShaderFunctionVisualizeNoise(VertexShaderOutput input) : COLOR0
{
	float2 flowVector;
	float noiseSample;
	GetFlowVectorAndNoiseSample(input.WorldPosition, flowVector, noiseSample);
	return float4(noiseSample.xxx, 1);
}

float4 PixelShaderFunctionVisualizeFlow(VertexShaderOutput input) : COLOR0
{
	float2 flowVector;
	float noiseSample;
	GetFlowVectorAndNoiseSample(input.WorldPosition, flowVector, noiseSample);
	if (UsingUnsignedFlowMap)
	{
		// Revert back
		flowVector = flowVector * 0.5  + 0.5;
	}
	return float4(flowVector, 1, 1);
}

technique Standard
{
    pass Pass1
    {
        VertexShader = compile vs_3_0 VertexShaderFunction();
        PixelShader = compile ps_3_0 PixelShaderFunction();
    }
}

technique VisualizeNoise
{
    pass Pass1
    {
        VertexShader = compile vs_3_0 VertexShaderFunction();
        PixelShader = compile ps_3_0 PixelShaderFunctionVisualizeNoise();
    }
}

technique VisualizeFlow
{
    pass Pass1
    {
        VertexShader = compile vs_3_0 VertexShaderFunction();
        PixelShader = compile ps_3_0 PixelShaderFunctionVisualizeFlow();
    }
}
